<?php

/**
.---------------------------------------------------------------------------.
|  Software: Apax Funcion - Ajax en php                                     |
|  @Version: 1.0                                                            |
| ------------------------------------------------------------------------- |
| Copyright © 2023, Diego Vazquez. All Rights Reserved.                     |
'---------------------------------------------------------------------------'
  Ultima fecha de modificación : 2023-08-11
______________________________________________________________________________________________________________________________________________________
ARGUMENTO    |  TIPO DE DATO  |         VALORES                 | DESCRIPCION
------------------------------------------------------------------------------------------------------------------------------------------------------
url          |  string        |                                 | url a la que se quiere hacer la petición.
type         |  string        |  "post", "get", "put", "delete" | tipo de petición::post::get::put::delete.
data         |  array         |                                 | datos a enviar.
dataType     |  string        |  "json", "array"                | parsea la respuesta de la petición::object::array. Puede generar una excepción si la respuesta no es un json valido
errorCodes   |  array         |  [203,302...]                   | puede usarse para añadir codigos de error http por debajo del estandar que es 400.
timeOut      |  integer       |  1,2,3...                       | puede usarse para definir el limite de tiempo de respuesta de la petición.

______________________________________________________________________________________________________________________________________________________
EVENTO          |  ARGUMENTOS                               |              RETORNO            | DESCRIPCION
------------------------------------------------------------------------------------------------------------------------------------------------------
onComplete      |                                           |                                 | se ejecuta siempre aún cuando haya errores.
onSuccess       |  response, http_code_status               |                                 | se ejecuta si la petición finaliza con exito.
onError         |  response, http_code_status, curl_errno   |                                 | se ejecuta si la petición falla.
onTimeout       |  response, http_code_status, curl_errno   |                                 | se ejecuta si la petición demora mas del tiempo establecido en el argumento "timeOut".
 */

if (!function_exists("Apax")) {
    function Apax(array $params)
    {   //funciones
        $is_json = function ($string) {
            json_decode($string);
            return json_last_error() === JSON_ERROR_NONE;
        };
        //argumentos requeridos
        if (!isset($params["url"])) throw new InvalidArgumentException('url is not defined');
        if (isset($params["url"]) && $params["url"] == "") throw new InvalidArgumentException('url can\'t be empty');
        if (!isset($params["type"])) throw new InvalidArgumentException('type is not defined');
        if (isset($params["type"]) && $params["type"] == "") throw new InvalidArgumentException('type can\'t be empty');
        if (!isset($params["data"])) $params["data"] = [];
        //argumentos opcionales
        if (!isset($params["dataType"])) $params["dataType"] = "";
        if (!isset($params["errorCodes"])) $params["errorCodes"] = [];
        if (!isset($params["timeOut"])) $params["timeOut"] = 30;
        //metodos
        switch (strtolower($params["type"])) {
            case "post": {
                    $query = http_build_query($params["data"]);
                    $ch    = curl_init();
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_HEADER, false);
                    curl_setopt($ch, CURLOPT_URL, $params["url"]);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $query);
                    curl_setopt($ch, CURLOPT_TIMEOUT, intval($params["timeOut"]));
                    break;
                }
            case "get": {
                    $query = http_build_query($params["data"]);
                    $ch    = curl_init($params["url"] . '?' . $query);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_HEADER, false);
                    curl_setopt($ch, CURLOPT_TIMEOUT, intval($params["timeOut"]));
                    break;
                }
            case "put": {
                    $query = \http_build_query($params["data"]);
                    $ch    = \curl_init();
                    \curl_setopt($ch, \CURLOPT_RETURNTRANSFER, true);
                    \curl_setopt($ch, \CURLOPT_HEADER, false);
                    \curl_setopt($ch, \CURLOPT_URL, $params["url"]);
                    \curl_setopt($ch, \CURLOPT_CUSTOMREQUEST, 'PUT');
                    \curl_setopt($ch, \CURLOPT_POSTFIELDS, $query);
                    \curl_setopt($ch, \CURLOPT_TIMEOUT, intval($params["timeOut"]));
                    break;
                }
            case "delete": {
                    $query = \http_build_query($params["data"]);
                    $ch    = \curl_init();
                    \curl_setopt($ch, \CURLOPT_RETURNTRANSFER, true);
                    \curl_setopt($ch, \CURLOPT_HEADER, false);
                    \curl_setopt($ch, \CURLOPT_URL, $params["url"]);
                    \curl_setopt($ch, \CURLOPT_CUSTOMREQUEST, 'DELETE');
                    \curl_setopt($ch, \CURLOPT_POSTFIELDS, $query);
                    \curl_setopt($ch, \CURLOPT_TIMEOUT, intval($params["timeOut"]));
                    break;
                }
        }
        //ejecuta la petición
        $response = curl_exec($ch);
        //http codigo de estatus
        $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        //codigo de error del curl
        $curl_errno = curl_errno($ch);
        //cierra el curl
        curl_close($ch);
        //existe algun error?
        $exists_error = $http_status >= 400 || in_array($http_status, $params["errorCodes"]);

        //callback complete
        if (isset($params["onComplete"]) && is_callable($params["onComplete"])) {
            $params["onComplete"]();
        }

        //parsea la respuesta de la petición a un objeto
        if (strtolower($params["dataType"]) == "json") {
            if ($is_json(strval($response))) {
                $response = json_decode(strval($response));
            } else {
                throw new JsonException('Response is not valid json');
            }
        }

        //parsea la respuesta de la petición a un arreglo asociativo
        if (strtolower($params["dataType"]) == "array") {
            if ($is_json(strval($response))) {
                $response = (array) json_decode(strval($response));
            } else {
                throw new JsonException('Response is not valid json');
            }
        }

        //callback timeout
        if ($curl_errno && in_array($curl_errno, [CURLE_OPERATION_TIMEDOUT, CURLE_OPERATION_TIMEOUTED]) && !$exists_error) {
            if (isset($params["onTimeout"]) && is_callable($params["onTimeout"])) {
                $params["onTimeout"]($response, $http_status, $curl_errno); //response, http status code, curl error
            }
        }

        //callback error
        if ($exists_error) {
            if (isset($params["onError"]) && is_callable($params["onError"])) {
                $params["onError"]($response, $http_status, $curl_errno); //response, http status code, curl error
            }
        }

        //callback success
        if (isset($params["onSuccess"]) && is_callable($params["onSuccess"]) && !$exists_error) {
            $params["onSuccess"]($response, $http_status); //response, http status code
        }

        //retorna la respuesta
        return $response;
    }
}
